﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using DataAccess;
using System.Collections.Generic;
using System.Data;

using System.Runtime.InteropServices;

/// <summary>
/// This class implements the IQLogic COM object so that we do not
/// have to go changing interfaces throughout the code -
/// </summary>
public class CIQEngine
{
    private BaseMaster m_BaseMstr = null;
    public string m_strErrorMessage;
    private string m_strVersion = String.Empty;

    private IQLogicLib.IIQLogicCOM m_iqLogic;
    public IQLogicLib.IIQLogicCOM GetIQLogic() { return m_iqLogic; }

    //const int nJSON_SCORE_TYPE = 10;
    //const int nJSON_FLAG_TYPE = 20;
    //const int nJSON_PROB_TYPE = 30;

    //const int nFLAG_TYPE = 10;
    //const int nPROBLEM_TYPE = 15;
    //const int nRESPONSE_TYPE = 20;
    //const int nLOGIC_TYPE = 30;

    const sbyte sbVarDelimit = 124; // | pipe character
    string strVarDelimit = "|";

    const sbyte sbLogicDelimit = 59; // ; semicolon
    string strLogicDelimit = ";";

    /// <summary>
    /// Constructor that takes BaseMaster
    /// </summary>
    /// <param name="bmstr"></param>
    public CIQEngine(BaseMaster bmstr)
	{
        m_BaseMstr = bmstr;
        m_strErrorMessage = "";

        GetCOMLib();

        //Load the data type definitions -
        //nFLAG_TYPE = m_iqLogic.GetFlagType();
        //nPROBLEM_TYPE = m_iqLogic.GetProblemType();
        //nLOGIC_TYPE = m_iqLogic.GetLogicType();
        //nRESPONSE_TYPE = m_iqLogic.GetResponseType();

        //Definitions for the type of retuned JSON
        //nJSON_FLAG_TYPE = 20;// m_iqLogic.JSON_FLAG_TYPE();
        //nJSON_SCORE_TYPE = 10;// m_iqLogic.JSON_SCORE_TYPE();
        //nJSON_PROB_TYPE = 30;// m_iqLogic.JSON_PROB_TYPE();

	}

    /// <summary>
    /// Get a reference to the com lib
    /// </summary>
    /// <returns></returns>
    private bool GetCOMLib()
    {
        try
        {
            m_iqLogic = new IQLogicLib.IQLogicCOMClass();
        }
        catch (COMException exp)
        {
            m_strErrorMessage = "~#(br /)#~~#(br /)#~GetCOMLib( ) - Unable to create the IQLogic COM component. ~#(br /)#~ Please contact your System Administrator. ~#(br /)#~~#(br /)#~" + exp.Message;
            throw new COMException(m_strErrorMessage);
            m_iqLogic = null;
            return false;
        }

        if (m_iqLogic == null)
        {
            m_strErrorMessage = "GetCOMLib( ) - Unable to create the IQLogic COM component";
            return false;
        }

        m_strErrorMessage = String.Empty;
        m_iqLogic.GetVersion(out m_strVersion);
       
        return true;
    }

    /// <summary>
    /// Add a SQL variable Item
    /// </summary>
    /// <param name="dsLV"></param>
    /// <returns></returns>
    public bool AddSQLVarItem(DataSet dsLV)
    {
        if (dsLV != null) 
        {
            foreach (DataTable dt in dsLV.Tables) 
            {
                foreach (DataRow dr in dt.Rows) 
                {
                    string strVariableTYPE = Convert.ToString((int) GetLOGIC_TYPE());

                    string strVariableName = String.Empty;
                    if (!dr.IsNull("identifier"))
                    {
                        strVariableName = dr["identifier"].ToString();
                    }
                    
                    string strVariableValue = String.Empty;
                    if(!dr.IsNull("var_value")){
                        strVariableValue = dr["var_value"].ToString();
                    }

                    string strKeyValue = strVariableTYPE + strVarDelimit + strVariableName + strVarDelimit + strVariableValue;

                    if (m_iqLogic != null)
                    {
                        m_iqLogic.Add(GetLOGIC_TYPE(), strKeyValue, sbVarDelimit);
                    }
                }
            }
        }

        return true;
    }

    /// <summary>
    /// Add a response variable item
    /// </summary>
    /// <param name="dsLV"></param>
    /// <returns></returns>
    public bool AddResponseVarItem(DataSet dsLV)
    {

        if (dsLV != null)
        {
            foreach (DataTable dt in dsLV.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    string strVariableTYPE = Convert.ToString((int)GetRESPONSE_TYPE());

                    string strVariableTID = String.Empty;
                    if (!dr.IsNull("TID"))
                    {
                        strVariableTID = dr["TID"].ToString();
                    }

                    string strVariableQID = String.Empty;
                    if (!dr.IsNull("QID"))
                    {
                        strVariableQID = dr["QID"].ToString();
                    }

                    string strVariableRID = String.Empty;
                    if (!dr.IsNull("RID"))
                    {
                        strVariableRID = dr["RID"].ToString();
                    }

                    string strVariableName = String.Empty;
                    if (!dr.IsNull("IDENTIFIER"))
                    {
                        strVariableName = dr["IDENTIFIER"].ToString();
                    }

                    //string strVariableValue = String.Empty;
                    //if (!dr.IsNull("RESPONSE_VALUE"))
                    //{
                    //    strVariableValue = dr["RESPONSE_VALUE"].ToString();
                    //}

                    long lResponseType = 0;
                    if (!dr.IsNull("RESPONSE_TYPE"))
                    {
                        lResponseType = Convert.ToInt64(dr["RESPONSE_TYPE"]);
                    }

                    //Set the value of the Response Variable
                    string strVariableValue = String.Empty;
                    if (lResponseType == 1 || lResponseType == 2)
                    {
                        //If the response ia a RADIO or CHECKBOX control
                        //we need to set the value of the response to 1 or 0
                        if (!dr.IsNull("RESPONSE_VALUE"))
                        {
                            if (dr["IS_SELECTED"].ToString() == "1")
                            {
                                strVariableValue = "1";
                            }
                            else
                            {
                                strVariableValue = "0";
                            }

                        }
                        else
                        {
                            strVariableValue = "0";
                        }
                    }
                    else
                    {
                        //is not a RADIO or CHECKBOX, get the value for the response
                        if (!dr.IsNull("RESPONSE_VALUE"))
                        {
                            strVariableValue = dr["RESPONSE_VALUE"].ToString();
                        }
                    }

                    string strVariableScore = String.Empty;
                    if (!dr.IsNull("SCORE_VALUE"))
                    {
                        strVariableScore = dr["SCORE_VALUE"].ToString();
                    }

                    string strSelected = String.Empty;
                    if (!dr.IsNull("IS_SELECTED"))
                    {
                        strSelected = dr["IS_SELECTED"].ToString();
                    }

                    string strKeyValue = strVariableTYPE + strVarDelimit + strVariableTID + strVarDelimit + strVariableQID + strVarDelimit + strVariableRID + strVarDelimit + strVariableName + strVarDelimit + strVariableValue + strVarDelimit + strVariableScore + strVarDelimit + strSelected;

                    if (m_iqLogic != null)
                    {
                        m_iqLogic.Add(GetRESPONSE_TYPE(), strKeyValue, sbVarDelimit);
                    }
                }
            }
        }

        return true;
    }
   
    /// <summary>
    /// Calculate an equation
    /// </summary>
    /// <param name="strEquation"></param>
    /// <returns></returns>
    public bool Calculate(string strEquation)
    {
        m_iqLogic.Calculate(strEquation, sbLogicDelimit);

        //Check for errors -
        int nError = m_iqLogic.GetErrorCode();
        if (nError > 0)
        {
            m_strErrorMessage = m_iqLogic.GetErrorDescription();
            return false;
        }

        return true;
    }

    /// <summary>
    /// Get logic JSon
    /// </summary>
    /// <param name="nType"></param>
    /// <returns></returns>
    public string GetJSon(int nType)
    {
        string strJson = "";
        m_iqLogic.GetJSon(nType, out strJson);
        return strJson;
    }

    /// <summary>
    /// Get the JSON score type from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetJSON_SCORE_TYPE()
    {
        int nType = (int) m_iqLogic.JSON_SCORE_TYPE();
        return nType;
    }

    /// <summary>
    /// Get the Json flag type from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetJSON_FLAG_TYPE()
    {
        int nType = (int) m_iqLogic.JSON_FLAG_TYPE();
        return nType;
    }

    /// <summary>
    /// Get the Json prob type definition from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetJSON_PROB_TYPE()
    {
        int nType =(int) m_iqLogic.JSON_PROB_TYPE();
        return nType;
    }

    /// <summary>
    /// Get the resposne type definition from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetRESPONSE_TYPE()
    {
        int nType = (int) m_iqLogic.GetResponseType();
        return nType;
    }

    /// <summary>
    /// Get the problem type definition from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetPROBLEM_TYPE()
    {
        int nType = (int) m_iqLogic.GetProblemType();
        return nType;
    }

    /// <summary>
    /// Get the logic type definition from the IQLogic
    /// </summary>
    /// <returns></returns>
    public int GetLOGIC_TYPE()
    {
        int nType = (int) m_iqLogic.GetLogicType();
        return nType;
    }

    //------------------------------------------------------------------
    /// <summary>
    /// Get the version from the IQLogic
    /// </summary>
    /// <returns></returns>
    public string GetVersion()
    {
        return m_strVersion;
    }


}